/** @file   route.h
 * @brief   Declaration of Route - class
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_ROUTE_H
#define H_WWW_ROUTE_H

#include <vector>
#include <iostream>
#include "vec2d.h"
#include "www_exceptions.h"


namespace WeWantWar {


/** @class  Route
 * @brief   Represents the route the WaypointController follows.
 * @author  Tomi Lamminsaari
 *
 * The Route consists of series of waypoints.
 *
 * @note    This class expects that the tilesize is 32x32 pixels.
 */
class Route
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** @struct Waypoint
   * @brief   Describes a single vertex of the route.
   */
  struct Waypoint {
    /** The center coordinate of this node. */
    eng2d::Vec2D  pos;
  };
  


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   * @param     routeName         Name of this Route. It's recommended that
   *                              this name is unique among all the Routes
   *                              you create.
   */
	Route( const std::string& routeName );


	/** Destructor
   */
	virtual ~Route();


	/** Copy constructor.
   * @param     rO                Reference to another Route
   */
  Route( const Route& rO );

	/** Assignment operator
   * @param     rO                Reference to another Route
   * @return    Reference to us.
   */
  Route& operator = ( const Route& rO );




  ///
  /// Methods
  /// =======

  /** Adds the given waypoint Node to this Route.
   * @param     rN                The Waypoint to be added
   */
  void addWaypoint( const Waypoint& rN );
  
  /** Sets the index'th WaypointNode.
   * @param     index             Index of the node to be set. If this is
   *                              invalid, this method does nothing.
   * @param     rN                New WaypointNode
   */
  void setWaypoint( int index, const Waypoint& rN );
  
  /** Loads the whole route from file.
   * @param     routefile         Name of the route-file.
   * @return    0 if successful. Nonzero if fails.
   */
  int loadRoute( const std::string& routefile );
  
  /** Reads the route data from given stream. This method expects that the
   * opening WEWANTWAR_ROUTE - tag has been read away from the begining of the
   * stream.
   * @param     rIn               The input stream to read from.
   * @return    0 if successful. Nonzero if fails.
   */
  int readRoute( std::istream& rIn );
  
  /** Clears all the waypoints of this Route.
   */
  void clear();


  ///
  /// Getter methods
  /// ==============
  
  /** Returns the name of this Route
   * @return    Name of this Route
   */
  std::string name() const;

  /** Returns the number of Waypoint nodes there are.
   * @return    Number of nodes this route has.
   */
  int nodeCount() const;
  
  /** Returns the index'th Waypoint.
   * @param     index             Index of the Waypoint being returned.
   * @return    Requested Waypoint-node.
   */
  const Waypoint& getWaypoint( int index ) const;
  
  /** Returns the distance from <code>rPos</code> to the index'th Node.
   * @param     rPos              A coordinate
   * @param     index             Index of the Node we calculate the distance
   *                              to.
   * @return    Distance from coordinate <code>rPos</code> to center of
   *            the index'th waypoint Node.
   */
  float distanceFromWaypoint( const eng2d::Vec2D& rPos, int index ) const;
  
  /** Returns the coordinate of requested Waypoint.
   * @param     index             Index of the Waypoint
   * @return    The coordinate where that Waypoint is located.
   */
  eng2d::Vec2D getWaypointPos( int index ) const;

protected:

  ///
  /// Protected methods
  /// =================
  
  /** Reads the node from given stream.
   * @param     rIn               The input stream
   * @return    0 if successful. Nonzero if fails.
   */
  int readNode( std::istream& rIn );
  
  /** Reads the node from given file. Here the read x and y coordinates
   * are block coordinates so they must be multiplied by 32 to get the
   * actual position of the node.
   * @param   aIn               The stream to read from.
   * @return  Nonzero on failure.
   */
  int readBlockNode( std::istream& aIn );
   

  ///
  /// Members
  /// =======

  /** This vector holds the waypoint Nodes added to this route */
  std::vector< Waypoint > m_waypoints;
  
  /** Name of this Route */
  std::string m_name;

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: route.h,v $
 * Revision 1.2  2006/03/29 22:27:19  lamminsa
 * readBlockNode() method added.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:12+02  lamminsa
 * Initial revision
 *
 */
 
